// ProfVault Extension Popup
// Uses APP_SETTINGS from config.js

const API_BASE_URL = APP_SETTINGS.API_BASE_URL;

// DOM Elements
const views = {
  login: document.getElementById('login-view'),
  connected: document.getElementById('connected-view'),
  filling: document.getElementById('filling-view'),
  results: document.getElementById('results-view'),
  error: document.getElementById('error-view'),
  cached: document.getElementById('cached-view')
};

const elements = {
  tokenInput: document.getElementById('token-input'),
  connectBtn: document.getElementById('connect-btn'),
  loginError: document.getElementById('login-error'),
  logoutBtn: document.getElementById('logout-btn'),
  fillFormBtn: document.getElementById('fill-form-btn'),
  doneBtn: document.getElementById('done-btn'),
  retryBtn: document.getElementById('retry-btn'),
  userName: document.getElementById('user-name'),
  userEmail: document.getElementById('user-email'),
  userAvatar: document.getElementById('user-avatar'),
  creditsBalance: document.getElementById('credits-balance'),
  profileCompleteness: document.getElementById('profile-completeness'),
  filledCount: document.getElementById('filled-count'),
  errorMessage: document.getElementById('error-message'),
  qaList: document.getElementById('qa-list'),
  copyAllBtn: document.getElementById('copy-all-btn'),
  // Cached view elements
  useCachedBtn: document.getElementById('use-cached-btn'),
  reanalyzeBtn: document.getElementById('reanalyze-btn'),
  cachedTime: document.getElementById('cached-time'),
  cachedFieldCount: document.getElementById('cached-field-count'),
  backFromCachedBtn: document.getElementById('back-from-cached-btn')
};

// State
let currentUser = null;
let currentCachedData = null; // Store current cached data for use

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  updateDynamicLinks();
  await checkAuthStatus();
  setupEventListeners();
  // Clean old cache entries on startup
  cleanOldCache();
});

// Update all links to use the correct base URL from config
function updateDynamicLinks() {
  const siteUrl = APP_SETTINGS.SITE_URL;

  // Update all links with data-path attribute
  document.querySelectorAll('a[data-path]').forEach(link => {
    link.href = siteUrl + link.dataset.path;
  });
}

// Store matched fields for Q&A display
let lastMatchedFields = [];

function setupEventListeners() {
  // Sign In button (opens ProfVault login page)
  const signInBtn = document.getElementById('signin-btn');
  if (signInBtn) {
    signInBtn.addEventListener('click', handleSignIn);
  }

  elements.connectBtn.addEventListener('click', handleConnect);
  elements.tokenInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') handleConnect();
  });
  elements.logoutBtn.addEventListener('click', handleLogout);
  elements.fillFormBtn.addEventListener('click', handleFillForm);
  elements.doneBtn.addEventListener('click', () => showView('connected'));
  elements.retryBtn.addEventListener('click', handleFillForm);
  elements.copyAllBtn?.addEventListener('click', copyAllQA);
  // Cached view event listeners
  elements.useCachedBtn?.addEventListener('click', handleUseCached);
  elements.reanalyzeBtn?.addEventListener('click', handleReanalyze);
  elements.backFromCachedBtn?.addEventListener('click', () => showView('connected'));
}

// Sign In - Opens auth page in new tab and monitors for token
async function handleSignIn() {
  const signInBtn = document.getElementById('signin-btn');
  const originalHTML = signInBtn.innerHTML;
  signInBtn.disabled = true;
  signInBtn.innerHTML = `
    <div class="spinner" style="width: 18px; height: 18px; border-width: 2px; margin: 0;"></div>
    Connecting...
  `;

  const authUrl = `${APP_SETTINGS.SITE_URL}/extension/auth`;

  // Open auth page in new tab
  const authTab = await chrome.tabs.create({ url: authUrl });

  // Monitor the tab for URL changes (when auth completes, URL will have token in hash)
  const checkInterval = setInterval(async () => {
    try {
      const tab = await chrome.tabs.get(authTab.id);

      // Check if URL contains token in hash
      if (tab.url && tab.url.includes('#token=')) {
        clearInterval(checkInterval);

        // Extract token from URL hash
        const urlHash = new URL(tab.url).hash;
        const token = urlHash.replace('#token=', '');

        if (token) {
          // Verify and store token
          const verified = await verifyAndStoreToken(token);

          if (verified) {
            // Close the auth tab
            try {
              await chrome.tabs.remove(authTab.id);
            } catch (e) {
              console.log('Tab already closed');
            }
          }
        }

        // Reset button
        signInBtn.disabled = false;
        signInBtn.innerHTML = originalHTML;
      }
    } catch (e) {
      // Tab was closed by user
      clearInterval(checkInterval);
      signInBtn.disabled = false;
      signInBtn.innerHTML = originalHTML;
    }
  }, 500); // Check every 500ms

  // Timeout after 5 minutes
  setTimeout(() => {
    clearInterval(checkInterval);
    signInBtn.disabled = false;
    signInBtn.innerHTML = originalHTML;
  }, 5 * 60 * 1000);
}

// Verify token and store if valid
async function verifyAndStoreToken(token) {
  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/auth/verify`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    const data = await response.json();

    if (data.success && data.valid) {
      await storeToken(token);
      currentUser = data.user;
      updateUserUI(data.user);
      await fetchProfile(token);
      showView('connected');
      return true;
    } else {
      showLoginError('Token verification failed. Please try again.');
      return false;
    }
  } catch (error) {
    console.error('Token verification failed:', error);
    showLoginError('Connection failed. Please try again.');
    return false;
  }
}

// View Management
function showView(viewName) {
  Object.values(views).forEach(view => view.classList.add('hidden'));
  views[viewName].classList.remove('hidden');
}

// Authentication
async function checkAuthStatus() {
  const token = await getStoredToken();

  if (!token) {
    showView('login');
    return;
  }

  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/auth/verify`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    const data = await response.json();

    if (data.success && data.valid) {
      currentUser = data.user;
      updateUserUI(data.user);
      await fetchProfile(token);

      // Check for cached results for current tab URL
      const hasCached = await checkCacheForCurrentTab();
      if (!hasCached) {
        showView('connected');
      }
    } else {
      await clearStoredToken();
      showView('login');
    }
  } catch (error) {
    console.error('Auth check failed:', error);
    showView('login');
  }
}

async function handleConnect() {
  const token = elements.tokenInput.value.trim();

  if (!token) {
    showLoginError('Please enter your API token');
    return;
  }

  elements.connectBtn.disabled = true;
  elements.connectBtn.textContent = 'Connecting...';
  elements.loginError.style.display = 'none';

  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/auth/verify`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    const data = await response.json();
    console.log('Verify response:', data);

    if (data.success && data.valid) {
      await storeToken(token);
      console.log('Token stored successfully');
      currentUser = data.user;
      updateUserUI(data.user);
      await fetchProfile(token);

      // Check for cached results for current tab URL
      const hasCached = await checkCacheForCurrentTab();
      if (!hasCached) {
        showView('connected');
      }
    } else {
      showLoginError('Invalid token. Please check and try again.');
    }
  } catch (error) {
    console.error('Connection failed:', error);
    showLoginError('Failed to connect. Is your server running?');
  } finally {
    elements.connectBtn.disabled = false;
    elements.connectBtn.textContent = 'Connect';
  }
}

function showLoginError(message) {
  elements.loginError.textContent = message;
  elements.loginError.style.display = 'block';
}


async function handleLogout() {
  await clearStoredToken();
  currentUser = null;
  location.reload();
}

// Profile
async function fetchProfile(token) {
  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/profile`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    const data = await response.json();

    if (data.success) {
      elements.creditsBalance.textContent = data.user.credits_balance || 0;
      // Calculate profile completeness from profile data
      const completeness = calculateProfileCompleteness(data.profile);
      elements.profileCompleteness.textContent = `${completeness}%`;
    }
  } catch (error) {
    console.error('Failed to fetch profile:', error);
  }
}

function calculateProfileCompleteness(profile) {
  let score = 0;
  if (profile.personal_info?.full_name) score += 20;
  if (profile.professional_summary) score += 15;
  if (profile.work_experiences?.length > 0) score += 25;
  if (profile.education?.length > 0) score += 15;
  if (Object.keys(profile.skills || {}).length > 0) score += 15;
  if (profile.projects?.length > 0) score += 10;
  return score;
}

// Form Filling
async function handleFillForm() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    const url = tab.url;

    console.log('ProfVault: Current tab URL:', url);

    // Check for cached results first
    const cached = await checkCacheForUrl(url);

    if (cached && cached.matchedFields && cached.matchedFields.length > 0) {
      // Show cached view with options
      console.log('ProfVault: Found cached results for URL');
      showCachedView(cached);
    } else {
      // No cache - do fresh analysis
      await performFreshFill(false);
    }
  } catch (error) {
    console.error('Form fill error:', error);
    showError('Could not fill the form. Make sure you are on a page with a form.');
  }
}

// Perform fresh form fill (calls API, charges credits, saves to cache)
async function performFreshFill(forceRefresh = false) {
  showView('filling');

  try {
    const token = await getStoredToken();

    // Send message to content script to detect and fill form
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    console.log('ProfVault: Performing fresh fill for URL:', tab.url);

    // Inject content script if not already present (into all frames)
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id, allFrames: true },
        files: ['content/content.js']
      });
      console.log('ProfVault: Content script injected successfully');
    } catch (injectError) {
      console.log('ProfVault: Script injection note:', injectError.message);
      // Script might already be injected, or page doesn't allow injection
    }

    // Give script a moment to initialize
    await new Promise(resolve => setTimeout(resolve, 300));

    // Try sending to all frames and collect responses
    let bestResponse = null;
    let totalFilledCount = 0;

    // First try main frame
    try {
      const mainResponse = await chrome.tabs.sendMessage(tab.id, {
        action: 'fillForm',
        token: token,
        apiBaseUrl: API_BASE_URL
      });
      if (mainResponse.success && mainResponse.filledCount > 0) {
        bestResponse = mainResponse;
        totalFilledCount += mainResponse.filledCount;
      } else if (!bestResponse) {
        bestResponse = mainResponse;
      }
    } catch (e) {
      console.log('Main frame did not respond:', e.message);
    }

    // Then try all frames
    try {
      const frames = await chrome.webNavigation.getAllFrames({ tabId: tab.id });
      for (const frame of frames) {
        if (frame.frameId === 0) continue; // Skip main frame, already tried

        try {
          const frameResponse = await chrome.tabs.sendMessage(tab.id, {
            action: 'fillForm',
            token: token,
            apiBaseUrl: API_BASE_URL
          }, { frameId: frame.frameId });

          if (frameResponse.success && frameResponse.filledCount > 0) {
            totalFilledCount += frameResponse.filledCount;
            if (!bestResponse || !bestResponse.success) {
              bestResponse = frameResponse;
            }
          } else if (!bestResponse) {
            bestResponse = frameResponse;
          }
        } catch (frameError) {
          // Frame might not have content script or no form
          console.log(`Frame ${frame.frameId} did not respond:`, frameError.message);
        }
      }
    } catch (navError) {
      console.log('Could not get frames:', navError.message);
    }

    // Use best response
    let errorMessage = 'No form fields found on this page';

    // Provide more helpful error messages
    if (!bestResponse) {
      const url = tab.url || '';
      if (url.startsWith('chrome://') || url.startsWith('chrome-extension://') || url.startsWith('about:')) {
        errorMessage = 'Cannot fill forms on browser internal pages.';
      } else if (url.includes('greenhouse.io') || url.includes('lever.co') || url.includes('workday') || url.includes('icims')) {
        errorMessage = 'Could not access form. Try clicking "Fill This Form" again after the page fully loads.';
      } else {
        errorMessage = 'No form fields found on this page.';
      }
    }

    const response = bestResponse || { success: false, error: errorMessage };

    if (response.success || totalFilledCount > 0) {
      elements.filledCount.textContent = totalFilledCount || response.filledCount || 0;

      // Display Q&A from matched fields
      if (response.matchedFields) {
        lastMatchedFields = response.matchedFields;
        displayQA(response.matchedFields);

        // Save to cache for future visits
        await saveToCache(tab.url, response.matchedFields, response.jobContext || {});
      }

      showView('results');
    } else {
      showError(response.error || 'Failed to fill form');
    }
  } catch (error) {
    console.error('Form fill error:', error);
    showError('Could not fill the form. Make sure you are on a page with a form.');
  }
}

// Display Q&A in the results view
function displayQA(matchedFields) {
  if (!elements.qaList) return;

  elements.qaList.innerHTML = '';

  const fieldsWithValues = matchedFields.filter(f => f.value && f.value.trim());

  if (fieldsWithValues.length === 0) {
    elements.qaList.innerHTML = '<div class="qa-empty">No answers generated</div>';
    return;
  }

  fieldsWithValues.forEach((field, index) => {
    const qaItem = document.createElement('div');
    qaItem.className = 'qa-item';
    qaItem.innerHTML = `
      <div class="qa-question">${escapeHtml(field.label || `Field ${index + 1}`)}</div>
      <div class="qa-answer" data-value="${escapeHtml(field.value)}" title="Click to copy">${escapeHtml(field.value)}</div>
    `;

    // Click to copy individual answer
    const answerEl = qaItem.querySelector('.qa-answer');
    answerEl.addEventListener('click', () => copyToClipboard(field.value, answerEl));

    elements.qaList.appendChild(qaItem);
  });
}

// Copy single answer to clipboard
async function copyToClipboard(text, element) {
  try {
    await navigator.clipboard.writeText(text);
    element.classList.add('copied');
    setTimeout(() => element.classList.remove('copied'), 1000);
  } catch (err) {
    console.error('Failed to copy:', err);
  }
}

// Copy all Q&A to clipboard
async function copyAllQA() {
  if (!lastMatchedFields.length) return;

  const text = lastMatchedFields
    .filter(f => f.value && f.value.trim())
    .map(f => `Q: ${f.label || 'Unknown'}\nA: ${f.value}`)
    .join('\n\n');

  try {
    await navigator.clipboard.writeText(text);
    elements.copyAllBtn.textContent = 'Copied!';
    setTimeout(() => {
      elements.copyAllBtn.textContent = 'Copy All';
    }, 1500);
  } catch (err) {
    console.error('Failed to copy all:', err);
  }
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
  if (!text) return '';
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// UI Updates
function updateUserUI(user) {
  elements.userName.textContent = user.full_name || 'User';
  elements.userEmail.textContent = user.email || '';
  elements.userAvatar.textContent = (user.full_name || user.email || 'U').charAt(0).toUpperCase();
  elements.creditsBalance.textContent = user.credits_balance || 0;
}

function showError(message) {
  elements.errorMessage.textContent = message;
  showView('error');
}

// Token Storage
async function getStoredToken() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['profvault_token'], (result) => {
      resolve(result.profvault_token || null);
    });
  });
}

async function storeToken(token) {
  return new Promise((resolve) => {
    chrome.storage.local.set({ profvault_token: token }, resolve);
  });
}

async function clearStoredToken() {
  return new Promise((resolve) => {
    chrome.storage.local.remove(['profvault_token'], resolve);
  });
}

// ===== Cache Management Functions =====

// Normalize URL (remove query params, trailing slashes)
function normalizeUrl(url) {
  try {
    const parsed = new URL(url);
    return `${parsed.origin}${parsed.pathname}`.replace(/\/$/, '');
  } catch (e) {
    return url;
  }
}

// Check if cache exists for URL
async function checkCacheForUrl(url) {
  const normalizedUrl = normalizeUrl(url);
  const data = await chrome.storage.local.get('formCache');
  const cache = data.formCache || {};
  return cache[normalizedUrl] || null;
}

// Save results to cache
async function saveToCache(url, matchedFields, jobContext) {
  const normalizedUrl = normalizeUrl(url);
  const data = await chrome.storage.local.get('formCache');
  const cache = data.formCache || {};

  cache[normalizedUrl] = {
    url: normalizedUrl,
    timestamp: Date.now(),
    matchedFields,
    jobContext
  };

  await chrome.storage.local.set({ formCache: cache });
  console.log('ProfVault: Saved results to cache for', normalizedUrl);
}

// Clear old cache entries (entries older than 7 days)
async function cleanOldCache() {
  const data = await chrome.storage.local.get('formCache');
  const cache = data.formCache || {};
  const sevenDaysAgo = Date.now() - (7 * 24 * 60 * 60 * 1000);
  let cleaned = false;

  for (const url in cache) {
    if (cache[url].timestamp < sevenDaysAgo) {
      delete cache[url];
      cleaned = true;
    }
  }

  if (cleaned) {
    await chrome.storage.local.set({ formCache: cache });
    console.log('ProfVault: Cleaned old cache entries');
  }
}

// Format relative time for display
function formatRelativeTime(timestamp) {
  const now = Date.now();
  const diff = now - timestamp;
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);
  const days = Math.floor(diff / 86400000);

  if (minutes < 1) return 'just now';
  if (minutes < 60) return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
  if (hours < 24) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
  return `${days} day${days > 1 ? 's' : ''} ago`;
}

// Show cached view with data
function showCachedView(cached) {
  currentCachedData = cached;

  // Update cached view elements
  if (elements.cachedTime) {
    elements.cachedTime.textContent = formatRelativeTime(cached.timestamp);
  }
  if (elements.cachedFieldCount) {
    const fieldCount = cached.matchedFields?.filter(f => f.value)?.length || 0;
    elements.cachedFieldCount.textContent = fieldCount;
  }

  showView('cached');
}

// Handle "Use Cached Results" button click
async function handleUseCached() {
  if (!currentCachedData || !currentCachedData.matchedFields) {
    showError('No cached data available');
    return;
  }

  showView('filling');

  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // Inject content script if needed
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id, allFrames: true },
        files: ['content/content.js']
      });
    } catch (injectError) {
      console.log('ProfVault: Script injection note:', injectError.message);
    }

    // Wait longer for content script to initialize
    await new Promise(resolve => setTimeout(resolve, 500));

    // Send cached fields to content script
    let totalFilledCount = 0;
    let lastError = null;
    let anyResponse = false;

    // Try main frame first
    try {
      console.log('ProfVault: Sending fillFromCache to main frame with', currentCachedData.matchedFields.length, 'fields');
      const mainResponse = await chrome.tabs.sendMessage(tab.id, {
        action: 'fillFromCache',
        cachedFields: currentCachedData.matchedFields
      });
      console.log('ProfVault: Main frame response:', mainResponse);
      anyResponse = true;
      if (mainResponse.success) {
        totalFilledCount += mainResponse.filledCount || 0;
      } else if (mainResponse.error) {
        lastError = mainResponse.error;
      }
    } catch (e) {
      console.log('Main frame did not respond:', e.message);
      lastError = 'Content script not responding. Try refreshing the page.';
    }

    // Try all frames
    try {
      const frames = await chrome.webNavigation.getAllFrames({ tabId: tab.id });
      console.log('ProfVault: Found', frames.length, 'frames');
      for (const frame of frames) {
        if (frame.frameId === 0) continue;

        try {
          const frameResponse = await chrome.tabs.sendMessage(tab.id, {
            action: 'fillFromCache',
            cachedFields: currentCachedData.matchedFields
          }, { frameId: frame.frameId });

          console.log('ProfVault: Frame', frame.frameId, 'response:', frameResponse);
          anyResponse = true;
          if (frameResponse.success) {
            totalFilledCount += frameResponse.filledCount || 0;
          } else if (frameResponse.error && !lastError) {
            lastError = frameResponse.error;
          }
        } catch (frameError) {
          // Frame might not have content script
          console.log('Frame', frame.frameId, 'error:', frameError.message);
        }
      }
    } catch (navError) {
      console.log('Could not get frames:', navError.message);
    }

    console.log('ProfVault: Total filled from cache:', totalFilledCount);

    if (totalFilledCount > 0) {
      elements.filledCount.textContent = totalFilledCount;
      lastMatchedFields = currentCachedData.matchedFields;
      displayQA(currentCachedData.matchedFields);
      showView('results');
    } else {
      // Show more specific error message
      const errorMsg = lastError || 'Could not fill form from cache. Try reanalyzing.';
      showError(errorMsg);
    }
  } catch (error) {
    console.error('Cache fill error:', error);
    showError('Could not fill the form from cache.');
  }
}

// Handle "Reanalyze" button click - force fresh analysis
async function handleReanalyze() {
  // Clear current cached data flag and do fresh fill
  currentCachedData = null;
  await performFreshFill(true); // true = force fresh (skip cache check)
}

// Check cache for current tab URL when popup opens
// Returns true if cached results were found and cached view was shown
async function checkCacheForCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab || !tab.url) return false;

    // Skip cache check for browser internal pages
    if (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://') || tab.url.startsWith('about:')) {
      return false;
    }

    const cached = await checkCacheForUrl(tab.url);

    if (cached && cached.matchedFields && cached.matchedFields.length > 0) {
      console.log('ProfVault: Found cached results on popup open for', tab.url);
      showCachedView(cached);
      return true;
    }

    return false;
  } catch (error) {
    console.error('Error checking cache for current tab:', error);
    return false;
  }
}
