// ProfVault Form Filler - Content Script

(function() {
  // Prevent multiple injections
  if (window.profVaultInjected) return;
  window.profVaultInjected = true;

  // Skip captcha and irrelevant iframes
  const currentUrl = window.location.href.toLowerCase();
  const skipPatterns = [
    'captcha',
    'recaptcha',
    'hcaptcha',
    'challenge',
    'grecaptcha',
    'gstatic.com/recaptcha'
  ];

  if (skipPatterns.some(pattern => currentUrl.includes(pattern))) {
    console.log('ProfVault: Skipping captcha/challenge iframe:', currentUrl);
    return;
  }

  // Debug logging
  const isInIframe = window !== window.top;
  console.log('ProfVault: Content script loaded', {
    url: window.location.href,
    isIframe: isInIframe,
    hasBody: !!document.body
  });

  // Listen for messages from popup
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'fillForm') {
      handleFillForm(request.token, request.apiBaseUrl)
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true; // Keep the message channel open for async response
    }

    if (request.action === 'fillFromCache') {
      handleFillFromCache(request.cachedFields)
        .then(result => sendResponse(result))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true; // Keep the message channel open for async response
    }

    if (request.action === 'detectForms') {
      const forms = detectForms();
      sendResponse({ success: true, formsCount: forms.length });
      return true;
    }
  });

  // Handle filling from cached data (no API call needed)
  async function handleFillFromCache(cachedFields) {
    try {
      if (!cachedFields || cachedFields.length === 0) {
        return { success: false, error: 'No cached fields to fill' };
      }

      console.log('ProfVault: Filling from cache with', cachedFields.length, 'cached fields');

      // Re-detect form fields on the current page
      let currentFields = detectFormFields();
      console.log('ProfVault: Detected', currentFields.length, 'current fields on page');

      // If no fields found initially, wait for dynamic content
      if (currentFields.length === 0) {
        console.log('ProfVault: No fields found, waiting for dynamic content...');
        await new Promise(resolve => setTimeout(resolve, 1000));
        currentFields = detectFormFields();
        console.log('ProfVault: After wait, detected', currentFields.length, 'fields');
      }

      if (currentFields.length === 0) {
        return { success: false, error: 'No form fields found on this page' };
      }

      // Create a map of cached values by various keys for quick lookup
      const cachedByLabel = {};
      const cachedById = {};
      const cachedByName = {};

      for (const cached of cachedFields) {
        if (cached.value) {
          if (cached.label) {
            const labelKey = cached.label.toLowerCase().trim();
            cachedByLabel[labelKey] = cached.value;
          }
          if (cached.id && !cached.id.startsWith('field_') && !cached.id.startsWith('custom_dropdown_')) {
            cachedById[cached.id] = cached.value;
          }
          if (cached.name) {
            cachedByName[cached.name] = cached.value;
          }
        }
      }

      console.log('ProfVault: Cached values - by label:', Object.keys(cachedByLabel).length,
                  'by id:', Object.keys(cachedById).length, 'by name:', Object.keys(cachedByName).length);

      // Match cached values to current fields
      const fieldsToFill = [];
      for (const field of currentFields) {
        let matchedValue = null;

        // Try to match by id first
        if (field.id && cachedById[field.id]) {
          matchedValue = cachedById[field.id];
          console.log('ProfVault: Matched by id:', field.id);
        }
        // Then by name
        else if (field.name && cachedByName[field.name]) {
          matchedValue = cachedByName[field.name];
          console.log('ProfVault: Matched by name:', field.name);
        }
        // Then by label (normalized)
        else if (field.label) {
          const labelKey = field.label.toLowerCase().trim();
          if (cachedByLabel[labelKey]) {
            matchedValue = cachedByLabel[labelKey];
            console.log('ProfVault: Matched by label:', field.label);
          } else {
            // Try partial label match
            for (const [cachedLabel, value] of Object.entries(cachedByLabel)) {
              if (labelKey.includes(cachedLabel) || cachedLabel.includes(labelKey)) {
                matchedValue = value;
                console.log('ProfVault: Matched by partial label:', field.label, '~', cachedLabel);
                break;
              }
            }
          }
        }

        if (matchedValue) {
          fieldsToFill.push({
            ...field,
            value: matchedValue
          });
        }
      }

      console.log('ProfVault: Matched', fieldsToFill.length, 'fields to fill from cache');

      if (fieldsToFill.length === 0) {
        return { success: false, error: 'Could not match cached values to current form fields' };
      }

      // Fill the matched fields
      const filledCount = await fillFormFields(fieldsToFill);

      return {
        success: filledCount > 0,
        filledCount: filledCount,
        fromCache: true,
        matchedFields: fieldsToFill
      };
    } catch (error) {
      console.error('ProfVault cache fill error:', error);
      return { success: false, error: 'Cache fill error: ' + error.message };
    }
  }

  // Extract job context from the page for better AI answers
  function extractJobContext() {
    const context = {
      page_url: window.location.href,
      page_title: document.title
    };

    // Try to extract job title from common patterns
    const jobTitleSelectors = [
      'h1', // Most sites put job title in h1
      '[data-test="job-title"]',
      '.job-title',
      '.posting-headline h2',
      '[class*="job-title"]',
      '[class*="JobTitle"]'
    ];

    for (const selector of jobTitleSelectors) {
      const el = document.querySelector(selector);
      if (el && el.textContent.trim()) {
        context.job_title = el.textContent.trim();
        break;
      }
    }

    // Try to extract company name
    const companySelectors = [
      '[data-test="company-name"]',
      '.company-name',
      '.posting-headline h1',
      '[class*="company-name"]',
      '[class*="CompanyName"]',
      '[class*="employer"]'
    ];

    for (const selector of companySelectors) {
      const el = document.querySelector(selector);
      if (el && el.textContent.trim()) {
        context.company = el.textContent.trim();
        break;
      }
    }

    // Fallback: try to extract from page title (e.g., "Software Engineer at Acme Corp")
    if (!context.job_title || !context.company) {
      const titleMatch = document.title.match(/^(.+?)\s+(?:at|@|-)\s+(.+?)(?:\s*[-|]|$)/i);
      if (titleMatch) {
        if (!context.job_title) context.job_title = titleMatch[1].trim();
        if (!context.company) context.company = titleMatch[2].trim();
      }
    }

    console.log('ProfVault: Extracted job context:', context);
    return context;
  }

  // Extract form HTML for backend AI processing
  function extractFormHtml() {
    // Try to find the main form container
    const formSelectors = [
      'form',
      '#application',
      '#app_form',
      '.application-form',
      '[id*="application"]',
      '[class*="application-form"]',
      '[class*="job-application"]',
      'main',
      '[role="main"]',
      '.content',
      '#content'
    ];

    let formElement = null;
    for (const selector of formSelectors) {
      const el = document.querySelector(selector);
      if (el) {
        // Check if this container has form fields
        const hasInputs = el.querySelectorAll('input, textarea, select').length > 0;
        if (hasInputs) {
          formElement = el;
          break;
        }
      }
    }

    // Fallback to body if no form container found
    if (!formElement) {
      formElement = document.body;
    }

    // Clone and clean the HTML to remove scripts, styles, and sensitive data
    const clone = formElement.cloneNode(true);

    // Remove scripts, styles, and other non-essential elements
    const removeSelectors = [
      'script', 'style', 'link', 'meta', 'noscript',
      'iframe[src*="captcha"]', 'iframe[src*="recaptcha"]',
      '[class*="captcha"]', '[id*="captcha"]',
      'svg', 'img', 'video', 'audio', 'canvas',
      '[style*="display: none"]', '[style*="display:none"]',
      '.hidden', '[hidden]'
    ];

    removeSelectors.forEach(selector => {
      clone.querySelectorAll(selector).forEach(el => el.remove());
    });

    // Get the cleaned HTML (limit size to avoid huge payloads)
    let html = clone.innerHTML;

    // Limit HTML size to ~50KB to avoid huge API payloads
    const maxLength = 50000;
    if (html.length > maxLength) {
      html = html.substring(0, maxLength) + '... [truncated]';
    }

    console.log('ProfVault: Extracted form HTML length:', html.length);
    return html;
  }

  // Main fill function - routes through background script to avoid CORS
  async function handleFillForm(token, apiBaseUrl) {
    try {
      // Detect all form fields on the page (with retry for dynamic loading)
      let fields = detectFormFields();
      console.log('ProfVault: Initial detected fields:', fields.length);

      // If no fields found, wait for dynamic content and retry
      // Job board sites (Greenhouse, Lever, etc.) often load forms dynamically
      if (fields.length === 0) {
        console.log('ProfVault: No fields found initially, waiting for dynamic content...');

        // First try: scroll to trigger lazy loading (some sites load form on scroll)
        window.scrollTo(0, document.body.scrollHeight / 2);
        await new Promise(resolve => setTimeout(resolve, 500));
        window.scrollTo(0, document.body.scrollHeight);
        await new Promise(resolve => setTimeout(resolve, 1000));

        fields = detectFormFields();
        console.log('ProfVault: After scroll + 1.5s, detected fields:', fields.length);

        // Second try: wait longer for AJAX-loaded content
        if (fields.length === 0) {
          await new Promise(resolve => setTimeout(resolve, 2000));
          fields = detectFormFields();
          console.log('ProfVault: After 3.5s total, detected fields:', fields.length);
        }

        // Third try: for very slow sites
        if (fields.length === 0) {
          await new Promise(resolve => setTimeout(resolve, 2000));
          fields = detectFormFields();
          console.log('ProfVault: After 5.5s total, detected fields:', fields.length);
        }
      }

      console.log('ProfVault: Field details:', fields.map(f => ({
        label: f.label,
        type: f.type,
        id: f.id,
        name: f.name,
        options: f.options?.length || 0
      })));

      if (fields.length === 0) {
        // Log debug info about the page
        const allInputs = document.querySelectorAll('input, textarea, select');
        const allForms = document.querySelectorAll('form');
        const allCustomSelects = document.querySelectorAll('[class*="select"], [class*="Select"], [class*="dropdown"]');
        const allIframes = document.querySelectorAll('iframe');

        console.log('ProfVault: Debug - Total input elements:', allInputs.length);
        console.log('ProfVault: Debug - Input details:', Array.from(allInputs).slice(0, 10).map(i => ({
          type: i.type, id: i.id, name: i.name, class: i.className?.substring(0, 50)
        })));
        console.log('ProfVault: Debug - Total forms:', allForms.length);
        console.log('ProfVault: Debug - Custom select elements:', allCustomSelects.length);
        console.log('ProfVault: Debug - Iframes on page:', allIframes.length);
        console.log('ProfVault: Debug - Document body exists:', !!document.body);
        console.log('ProfVault: Debug - URL:', window.location.href);

        // Check for common Greenhouse/job board form containers
        const greenhouseApp = document.querySelector('#application, #app_form, .application-form, [id*="application"]');
        const applyButton = document.querySelector(
          '[class*="apply"], button[data-qa*="apply"], a[href*="apply"], ' +
          '#apply_button, .apply-button, [class*="ApplyButton"], ' +
          'a[href*="/apply"], button[class*="submit-application"]'
        );

        // Check if this is a job listing page (not the application form)
        const isJobListingPage = window.location.href.includes('job-boards.greenhouse.io') ||
          document.querySelector('.job-description, .job-details, .posting-headline, [class*="job-description"]');
        const hasApplyLink = document.querySelector('a[href*="boards.greenhouse.io"], a[href*="/apply"]');

        if (greenhouseApp) {
          console.log('ProfVault: Debug - Found application container but no fields detected');
        }

        // If on a job listing page with an Apply link, suggest clicking it
        if (isJobListingPage && (applyButton || hasApplyLink)) {
          console.log('ProfVault: Debug - This is a job listing page, not the application form');
          return {
            success: false,
            error: 'This appears to be a job listing page. Click the "Apply" button to go to the application form, then try again.'
          };
        }

        if (applyButton && !greenhouseApp) {
          console.log('ProfVault: Debug - Found Apply button - user may need to click it first');
          return { success: false, error: 'Please click the "Apply" button first to load the application form, then try again.' };
        }

        return { success: false, error: 'No form fields found on this page. Make sure the application form is visible.' };
      }

      // Extract job context for AI-powered answers
      const jobContext = extractJobContext();

      // Extract form HTML for better AI context
      const formHtml = extractFormHtml();

      // Send fields to background script for API matching (avoids CORS)
      console.log('ProfVault: Sending to background script...');
      let data;
      try {
        data = await chrome.runtime.sendMessage({
          action: 'matchFields',
          token: token,
          fields: fields,
          jobContext: jobContext,
          formHtml: formHtml
        });
      } catch (msgError) {
        console.error('ProfVault: Message send failed:', msgError);
        return { success: false, error: 'Extension communication error. Try reloading the extension.' };
      }
      console.log('ProfVault: Response from background:', data);

      if (!data) {
        return { success: false, error: 'No response from extension. Try reloading it.' };
      }

      if (!data.success) {
        return { success: false, error: data.error || 'Failed to match fields' };
      }

      // Log matched fields with values
      console.log('ProfVault: Matched fields from API:');
      data.matched_fields.forEach((f, i) => {
        console.log(`  ${i}: label="${f.label}" value="${f.value}" type="${f.type}" confidence=${f.confidence}`);
      });

      // Fill the form with matched values (async - fill custom dropdowns sequentially)
      const filledCount = await fillFormFields(data.matched_fields);

      return {
        success: true,
        filledCount: filledCount,
        matchedFields: data.matched_fields  // Return for Q&A display
      };
    } catch (error) {
      console.error('ProfVault fill error:', error);
      return { success: false, error: 'Extension error: ' + error.message };
    }
  }

  // Detect all fillable form fields on the page
  function detectFormFields() {
    const fields = [];
    let fieldIndex = 0;

    // First, detect custom dropdown/combobox elements (React, Vue, etc.)
    // We do this first so we can skip their internal inputs
    const customDropdowns = detectCustomDropdowns();
    const customDropdownInputIds = new Set();

    customDropdowns.forEach((dropdown) => {
      const fieldInfo = {
        id: dropdown.id || `custom_dropdown_${fieldIndex}`,
        name: dropdown.name || '',
        type: 'custom_dropdown',
        label: dropdown.label,
        placeholder: dropdown.placeholder || '',
        required: dropdown.required || false,
        index: fieldIndex,
        options: dropdown.options,
        customElement: true,
        selector: dropdown.selector
      };
      fields.push(fieldInfo);
      fieldIndex++;

      // Track input IDs that are part of custom dropdowns
      if (dropdown.inputIds) {
        dropdown.inputIds.forEach(id => customDropdownInputIds.add(id));
      }
    });

    // Detect native form elements
    const inputs = document.querySelectorAll('input, textarea, select');

    // Skip elements that are part of cookie consent banners, navigation, or other non-form UI
    // NOTE: Be careful not to skip form inputs - keep selectors specific
    const skipSelectors = [
      '[class*="cookie-consent"]',
      '[class*="cookie-banner"]',
      '[class*="CookieConsent"]',
      '[id*="cookie-consent"]',
      '[id*="cky-consent"]',
      '[class*="consent-banner"]',
      '[class*="gdpr-banner"]',
      '[class*="privacy-banner"]'
    ];

    function shouldSkipInput(element) {
      for (const selector of skipSelectors) {
        try {
          if (element.closest(selector)) {
            return true;
          }
        } catch (e) {
          // Invalid selector, skip it
        }
      }
      return false;
    }

    inputs.forEach((input) => {
      // Skip hidden, submit, button, file inputs
      if (input.type === 'hidden' ||
          input.type === 'submit' ||
          input.type === 'button' ||
          input.type === 'file' ||
          input.type === 'image' ||
          input.type === 'reset') {
        return;
      }

      // Skip password fields
      if (input.type === 'password') {
        return;
      }

      // Skip captcha-related fields
      const inputName = (input.name || '').toLowerCase();
      const inputId = (input.id || '').toLowerCase();
      if (inputName.includes('captcha') || inputName.includes('recaptcha') || inputName.includes('hcaptcha') ||
          inputId.includes('captcha') || inputId.includes('recaptcha') || inputId.includes('hcaptcha')) {
        return;
      }

      // Skip cookie banners, navigation, and other non-form UI
      if (shouldSkipInput(input)) {
        return;
      }

      // Skip inputs that are part of custom dropdowns we already detected
      if (input.id && customDropdownInputIds.has(input.id)) {
        return;
      }

      // Check if this input is inside a custom dropdown component
      if (isInsideCustomDropdown(input)) {
        return;
      }

      const fieldInfo = {
        id: input.id || `field_${fieldIndex}`,
        name: input.name || '',
        type: getFieldType(input),
        label: findLabel(input),
        placeholder: input.placeholder || '',
        autocomplete: input.autocomplete || '',
        required: input.required,
        index: fieldIndex
      };

      // For select elements, include available options
      if (input.tagName === 'SELECT') {
        fieldInfo.options = getSelectOptions(input);
      }

      // For checkbox groups, include the value/label
      if (input.type === 'checkbox') {
        fieldInfo.checkboxValue = input.value || '';
        fieldInfo.checkboxLabel = findLabel(input) || input.value || '';
        // Try to find the group label for checkbox groups
        fieldInfo.groupLabel = findCheckboxGroupLabel(input);
      }

      // For radio buttons, include the value/label
      if (input.type === 'radio') {
        fieldInfo.radioValue = input.value || '';
        fieldInfo.radioLabel = findLabel(input) || input.value || '';
        fieldInfo.groupLabel = findRadioGroupLabel(input);
      }

      fields.push(fieldInfo);
      fieldIndex++;
    });

    return fields;
  }

  // Check if an input is inside a custom dropdown component
  function isInsideCustomDropdown(input) {
    // Check for react-select, vue-select, or similar patterns
    const customDropdownPatterns = [
      '[class*="select__input"]',
      '[class*="select-input"]',
      '[class*="dropdown-input"]',
      '[class*="combobox-input"]',
      '[class*="autocomplete-input"]',
      '[class*="searchable-select"]'
    ];

    // Check if input matches any pattern
    for (const pattern of customDropdownPatterns) {
      if (input.matches(pattern)) {
        return true;
      }
    }

    // Check parent elements for dropdown indicators
    const parent = input.closest('[class*="select"], [class*="dropdown"], [class*="combobox"]');
    if (parent) {
      // Check for dropdown indicators (clear button, dropdown arrow, etc.)
      const hasDropdownIndicator = parent.querySelector('[class*="indicator"], [class*="arrow"], [class*="caret"], svg');
      const hasListbox = parent.querySelector('[role="listbox"], [role="option"], [class*="option"], [class*="menu"]');

      if (hasDropdownIndicator || hasListbox) {
        return true;
      }

      // Check for aria attributes indicating dropdown
      if (parent.querySelector('[aria-haspopup="listbox"], [aria-expanded], [role="combobox"]')) {
        return true;
      }
    }

    // Check if input has combobox role or is inside one
    if (input.getAttribute('role') === 'combobox' || input.closest('[role="combobox"]')) {
      return true;
    }

    return false;
  }

  // Get the field type, normalizing select elements
  function getFieldType(input) {
    if (input.tagName === 'SELECT') {
      return 'select';
    }
    if (input.tagName === 'TEXTAREA') {
      return 'textarea';
    }
    return input.type || 'text';
  }

  // Get options from a select element
  function getSelectOptions(select) {
    const options = [];
    for (const option of select.options) {
      if (option.value !== '' || option.text.trim() !== '') {
        options.push({
          value: option.value,
          text: option.text.trim()
        });
      }
    }
    return options;
  }

  // Find the group label for checkbox groups
  function findCheckboxGroupLabel(checkbox) {
    // Look for a common parent with a label/legend
    const fieldset = checkbox.closest('fieldset');
    if (fieldset) {
      const legend = fieldset.querySelector('legend');
      if (legend) return legend.textContent.trim();
    }

    // Look for a parent div with a label-like element
    const parent = checkbox.closest('[class*="group"], [class*="field"], [role="group"]');
    if (parent) {
      const groupLabel = parent.querySelector('label:first-of-type, .label, [class*="label"]:not(label)');
      if (groupLabel && !groupLabel.contains(checkbox)) {
        return groupLabel.textContent.trim();
      }
    }

    return '';
  }

  // Find the group label for radio button groups
  function findRadioGroupLabel(radio) {
    // Same logic as checkbox groups
    return findCheckboxGroupLabel(radio);
  }

  // Detect custom dropdown elements (non-native selects)
  function detectCustomDropdowns() {
    const dropdowns = [];
    const processedElements = new Set();

    // Skip elements that are part of cookie consent banners, navigation, or other non-form UI
    // NOTE: Be careful not to skip dropdown menus which are part of forms!
    const skipSelectors = [
      '[class*="cookie"]',
      '[class*="Cookie"]',
      '[id*="cookie"]',
      '[id*="cky"]',
      '[class*="consent"]',
      '[class*="gdpr"]',
      '[class*="privacy"]',
      'nav:not([class*="form"])',
      '[role="navigation"]',
      'header.site-header',
      'header.main-header',
      '[class*="site-header"]',
      '[class*="main-header"]',
      'footer:not(.application-footer)',
      '[class*="nav-bar"]',
      '[class*="navbar"]',
      '[class*="site-footer"]',
      '[class*="main-footer"]'
    ];

    function shouldSkipElement(element) {
      // Check if element or its ancestors match skip selectors
      for (const selector of skipSelectors) {
        if (element.closest(selector)) {
          return true;
        }
      }
      return false;
    }

    // Pattern 1: React-select and similar library patterns
    // These have a container with class containing "select" and internal structure with indicators
    const selectContainers = document.querySelectorAll(
      '[class*="select"]:not(select), [class*="Select"]:not(select), ' +
      '[class*="dropdown"]:not(select), [class*="Dropdown"]:not(select), ' +
      '[class*="combobox"], [class*="Combobox"]'
    );

    selectContainers.forEach((container, index) => {
      // Skip if already processed or is a native select
      if (processedElements.has(container) || container.tagName === 'SELECT') {
        return;
      }

      // Skip cookie banners, navigation, and other non-form UI
      if (shouldSkipElement(container)) {
        return;
      }

      // Skip if this element is INSIDE another container we've already processed
      // This prevents nested divs from being detected as separate dropdowns
      let isNested = false;
      for (const processed of processedElements) {
        if (processed.contains(container) && processed !== container) {
          isNested = true;
          break;
        }
      }
      if (isNested) {
        return;
      }

      // Skip if a parent element has the same class pattern (we want the outermost one)
      const parentWithSamePattern = container.parentElement?.closest(
        '[class*="select"]:not(select), [class*="Select"]:not(select), ' +
        '[class*="dropdown"]:not(select), [class*="Dropdown"]:not(select)'
      );
      if (parentWithSamePattern && !processedElements.has(parentWithSamePattern)) {
        // Skip this one, we'll process the parent instead
        return;
      }

      // Check if this looks like a custom select component
      const hasIndicator = container.querySelector(
        '[class*="indicator"], [class*="Indicator"], ' +
        '[class*="arrow"], [class*="Arrow"], ' +
        '[class*="caret"], [class*="chevron"], svg'
      );
      const hasControl = container.querySelector('[class*="control"], [class*="Control"]');
      const hasInput = container.querySelector('input[type="text"], input:not([type])');

      // Must have indicator or control to be considered a dropdown
      if (!hasIndicator && !hasControl) {
        return;
      }

      // Get the label for this dropdown
      const label = findCustomDropdownLabel(container);
      if (!label) {
        return; // Skip if we can't identify what this dropdown is for
      }

      // Find any internal inputs to track
      const inputIds = [];
      const inputs = container.querySelectorAll('input');
      inputs.forEach(inp => {
        if (inp.id) inputIds.push(inp.id);
      });

      // Try to get current value from the displayed text
      const valueContainer = container.querySelector(
        '[class*="single-value"], [class*="SingleValue"], ' +
        '[class*="value"], [class*="Value"], ' +
        '[class*="placeholder"], [class*="Placeholder"]'
      );
      const currentValue = valueContainer?.textContent?.trim() || '';

      // Get options - may need to click to reveal them
      let options = getCustomDropdownOptions(container);

      // Create the dropdown entry
      const dropdown = {
        id: container.id || `react_select_${index}`,
        name: hasInput?.name || '',
        label: label,
        placeholder: currentValue || hasInput?.placeholder || '',
        required: container.hasAttribute('aria-required') ||
                  container.querySelector('[aria-required="true"]') !== null,
        options: options,
        selector: generateUniqueSelector(container),
        inputIds: inputIds
      };

      dropdowns.push(dropdown);
      processedElements.add(container);
      container.dataset.profvaultProcessed = 'true';

      // Also mark all children as processed to prevent them from being detected separately
      container.querySelectorAll('*').forEach(child => {
        processedElements.add(child);
      });
    });

    // Pattern 2: Elements with combobox or listbox roles
    const comboboxes = document.querySelectorAll('[role="combobox"], [aria-haspopup="listbox"]');

    comboboxes.forEach((element, index) => {
      // Skip if already processed or is native select
      if (processedElements.has(element) || element.tagName === 'SELECT' || element.closest('select')) {
        return;
      }

      // Skip cookie banners, navigation, and other non-form UI
      if (shouldSkipElement(element)) {
        return;
      }

      // Skip if parent was already processed
      if (element.closest('[data-profvault-processed="true"]')) {
        return;
      }

      const label = findCustomDropdownLabel(element);
      const dropdown = {
        id: element.id || `combobox_${index}`,
        name: element.getAttribute('name') || '',
        label: label,
        placeholder: element.getAttribute('placeholder') || element.textContent?.trim() || '',
        required: element.hasAttribute('aria-required') || element.hasAttribute('required'),
        options: getCustomDropdownOptions(element),
        selector: generateUniqueSelector(element),
        inputIds: []
      };

      // Only include if we found a label
      if (dropdown.label) {
        dropdowns.push(dropdown);
        processedElements.add(element);
        element.dataset.profvaultProcessed = 'true';
      }
    });

    // Pattern 3: Button-based dropdowns
    const buttonDropdowns = document.querySelectorAll(
      'button[aria-haspopup="true"], button[aria-expanded], ' +
      '[class*="dropdown-trigger"], [class*="select-button"]'
    );

    buttonDropdowns.forEach((element, index) => {
      if (processedElements.has(element) || element.closest('[data-profvault-processed="true"]')) {
        return;
      }

      // Skip cookie banners, navigation, and other non-form UI
      if (shouldSkipElement(element)) {
        return;
      }

      const listboxId = element.getAttribute('aria-controls') || element.getAttribute('aria-owns');
      let options = [];

      if (listboxId) {
        const listbox = document.getElementById(listboxId);
        if (listbox) {
          options = getOptionsFromListbox(listbox);
        }
      }

      if (options.length === 0) {
        const parent = element.closest('[class*="select"], [class*="dropdown"]');
        if (parent) {
          const optionContainer = parent.querySelector('[role="listbox"], [class*="options"], [class*="menu"]');
          if (optionContainer) {
            options = getOptionsFromListbox(optionContainer);
          }
        }
      }

      const label = findCustomDropdownLabel(element);
      if (options.length > 0 || label) {
        dropdowns.push({
          id: element.id || `button_dropdown_${index}`,
          name: '',
          label: label,
          placeholder: element.textContent?.trim() || '',
          required: element.hasAttribute('aria-required'),
          options: options,
          selector: generateUniqueSelector(element),
          inputIds: []
        });
        processedElements.add(element);
        element.dataset.profvaultProcessed = 'true';
      }
    });

    return dropdowns;
  }

  // Clean up label text by removing placeholder values and extra whitespace
  function cleanLabelText(text) {
    if (!text) return '';

    // Remove common placeholder patterns
    let cleaned = text
      .replace(/select\.\.\.$/i, '')     // Remove "Select..."
      .replace(/select$/i, '')            // Remove "Select"
      .replace(/choose\.\.\.$/i, '')      // Remove "Choose..."
      .replace(/--\s*$/i, '')             // Remove "--"
      .replace(/\s*\*\s*$/, '')           // Remove trailing asterisk (required indicator)
      .trim();

    return cleaned;
  }

  // Find label for custom dropdown
  function findCustomDropdownLabel(element) {
    // SPECIAL CASE: Phone input country selector
    // Check if this is inside a phone input container
    const phoneInputContainer = element.closest('.phone-input, [class*="phone-input"], [class*="PhoneInput"], fieldset.phone-input');
    if (phoneInputContainer) {
      // Check if this is the country code dropdown (not the phone number input)
      const isCountrySelector = element.closest('.phone-input__country, [class*="country"], .iti__country-container, [class*="CountrySelect"]');
      if (isCountrySelector) {
        return 'Phone Country Code';
      }
      // If it's inside phone-input but not country selector, it might be the phone field itself
      return 'Phone';
    }

    // Check aria-labelledby
    const labelledBy = element.getAttribute('aria-labelledby');
    if (labelledBy) {
      const label = document.getElementById(labelledBy);
      if (label) return cleanLabelText(label.textContent);
    }

    // Check aria-label
    const ariaLabel = element.getAttribute('aria-label');
    if (ariaLabel) return cleanLabelText(ariaLabel);

    // Check for IMMEDIATE parent field container with a label (not ancestors)
    // Start from the direct parent and go up only a few levels
    let currentParent = element.parentElement;
    let depth = 0;
    const maxDepth = 5; // Don't go too far up the DOM

    while (currentParent && depth < maxDepth) {
      // Check if this is a field container
      const isFieldContainer =
        currentParent.className?.match?.(/\b(field|form-group|form-field|question|input-group|FormField)\b/i) ||
        currentParent.hasAttribute('data-field');

      if (isFieldContainer) {
        // Look for a DIRECT label child (not nested deep inside)
        const directLabels = currentParent.querySelectorAll(':scope > label, :scope > .label, :scope > [class*="label"]:not([class*="input"])');
        for (const label of directLabels) {
          if (!label.contains(element) && label.textContent.trim()) {
            const labelText = cleanLabelText(label.textContent);
            // Make sure this label is not too far from our element
            if (labelText && labelText.length < 200) {
              return labelText;
            }
          }
        }

        // Also check for fieldset > legend
        if (currentParent.tagName === 'FIELDSET') {
          const legend = currentParent.querySelector(':scope > legend');
          if (legend) {
            return cleanLabelText(legend.textContent);
          }
        }
      }

      currentParent = currentParent.parentElement;
      depth++;
    }

    // Check immediate preceding siblings only
    let sibling = element.previousElementSibling;
    let siblingCount = 0;
    const maxSiblings = 3;

    while (sibling && siblingCount < maxSiblings) {
      if (sibling.tagName === 'LABEL' ||
          sibling.classList?.contains?.('label')) {
        return cleanLabelText(sibling.textContent);
      }
      // Check for text content that looks like a question
      const text = sibling.textContent?.trim();
      if (text && text.includes('?') && text.length < 200) {
        return cleanLabelText(text);
      }
      sibling = sibling.previousElementSibling;
      siblingCount++;
    }

    return '';
  }

  // Get options from a custom dropdown
  function getCustomDropdownOptions(element) {
    const options = [];

    // Check aria-controls or aria-owns for associated listbox
    const listboxId = element.getAttribute('aria-controls') || element.getAttribute('aria-owns');
    if (listboxId) {
      const listbox = document.getElementById(listboxId);
      if (listbox) {
        return getOptionsFromListbox(listbox);
      }
    }

    // Check if element itself contains options
    const optionElements = element.querySelectorAll('[role="option"], [class*="option"]');
    optionElements.forEach((opt) => {
      const value = opt.getAttribute('data-value') || opt.getAttribute('value') || opt.textContent.trim();
      const text = opt.textContent.trim();
      if (text && !options.some(o => o.text === text)) {
        options.push({ value, text });
      }
    });

    // Look in sibling/nearby elements for options
    if (options.length === 0) {
      const parent = element.closest('[class*="select"], [class*="dropdown"], [class*="field"]');
      if (parent) {
        const listbox = parent.querySelector('[role="listbox"], [class*="options"], [class*="menu"], ul');
        if (listbox && listbox !== element) {
          return getOptionsFromListbox(listbox);
        }
      }
    }

    return options;
  }

  // Extract options from a listbox element
  function getOptionsFromListbox(listbox) {
    const options = [];
    const optionElements = listbox.querySelectorAll('[role="option"], li, [class*="option"]');

    optionElements.forEach((opt) => {
      const value = opt.getAttribute('data-value') || opt.getAttribute('value') || opt.textContent.trim();
      const text = opt.textContent.trim();
      if (text && !options.some(o => o.text === text)) {
        options.push({ value, text });
      }
    });

    return options;
  }

  // Generate a unique CSS selector for an element
  function generateUniqueSelector(element) {
    // Helper to safely escape CSS identifiers
    const escapeId = (id) => {
      try {
        return CSS.escape(id);
      } catch (e) {
        // Fallback: escape special characters manually
        return id.replace(/([^\w-])/g, '\\$1');
      }
    };

    const escapeClass = (cls) => {
      try {
        return CSS.escape(cls);
      } catch (e) {
        return cls.replace(/([^\w-])/g, '\\$1');
      }
    };

    if (element.id) {
      return `#${escapeId(element.id)}`;
    }

    // Build path from element to a unique ancestor
    const path = [];
    let current = element;

    while (current && current !== document.body) {
      let selector = current.tagName.toLowerCase();

      if (current.id) {
        selector = `#${escapeId(current.id)}`;
        path.unshift(selector);
        break;
      }

      if (current.className && typeof current.className === 'string') {
        const classes = current.className.trim().split(/\s+/)
          .filter(c => c && !c.startsWith('profvault'))
          .slice(0, 2)
          .map(c => escapeClass(c));
        if (classes.length > 0) {
          selector += '.' + classes.join('.');
        }
      }

      // Add nth-child if needed for uniqueness
      const parent = current.parentElement;
      if (parent) {
        const siblings = Array.from(parent.children).filter(c => c.tagName === current.tagName);
        if (siblings.length > 1) {
          const index = siblings.indexOf(current) + 1;
          selector += `:nth-child(${index})`;
        }
      }

      path.unshift(selector);
      current = current.parentElement;
    }

    return path.join(' > ');
  }

  // Find the label for an input element
  function findLabel(input) {
    // Check for associated label by 'for' attribute
    if (input.id) {
      try {
        const label = document.querySelector(`label[for="${input.id}"]`);
        if (label) return label.textContent.trim();
      } catch (e) {
        // ID might contain special characters
        try {
          const label = document.querySelector(`label[for="${CSS.escape(input.id)}"]`);
          if (label) return label.textContent.trim();
        } catch (e2) {
          // Ignore selector errors
        }
      }
    }

    // Check for parent label
    const parentLabel = input.closest('label');
    if (parentLabel) {
      // Get text content excluding the input itself
      const clone = parentLabel.cloneNode(true);
      const inputsInClone = clone.querySelectorAll('input, textarea, select');
      inputsInClone.forEach(el => el.remove());
      return clone.textContent.trim();
    }

    // Check for preceding sibling that looks like a label
    let sibling = input.previousElementSibling;
    while (sibling) {
      if (sibling.tagName === 'LABEL' ||
          sibling.classList.contains('label') ||
          sibling.classList.contains('form-label')) {
        return sibling.textContent.trim();
      }
      // Check for Lever-style question text (div with question text)
      // Lever uses various class patterns for question labels
      if (sibling.classList.contains('application-label') ||
          sibling.classList.contains('question-label') ||
          sibling.getAttribute('data-qa')?.includes('label')) {
        return sibling.textContent.trim();
      }
      // Check if sibling contains a label element
      const innerLabel = sibling.querySelector('label');
      if (innerLabel) {
        return innerLabel.textContent.trim();
      }
      sibling = sibling.previousElementSibling;
    }

    // Check for aria-label
    if (input.getAttribute('aria-label')) {
      return input.getAttribute('aria-label');
    }

    // Check parent container for label (Lever puts questions in containers)
    const questionContainer = input.closest('.application-question, .custom-question, .form-group, .question, [class*="question"]');
    if (questionContainer) {
      // Look for label or question text at the beginning of container
      const labelEl = questionContainer.querySelector('label, .application-label, .question-label, [class*="label"]:not(input)');
      if (labelEl && !labelEl.contains(input)) {
        return labelEl.textContent.trim();
      }
      // Fallback: check first text-containing element that's not the input
      const firstText = questionContainer.querySelector('p, span, div');
      if (firstText && !firstText.contains(input) && firstText.textContent.trim().length < 500) {
        return firstText.textContent.trim();
      }
    }

    // Check parent for label-like elements
    const parent = input.parentElement;
    if (parent) {
      const labelElement = parent.querySelector('.label, .form-label, [class*="label"]');
      if (labelElement && labelElement !== input) {
        return labelElement.textContent.trim();
      }
    }

    return '';
  }

  // Detect forms on the page
  function detectForms() {
    return document.querySelectorAll('form');
  }

  // Fill form fields with matched values
  async function fillFormFields(matchedFields) {
    let filledCount = 0;

    for (let index = 0; index < matchedFields.length; index++) {
      const field = matchedFields[index];
      console.log(`ProfVault: Processing field ${index}: label="${field.label}" type="${field.type}" value="${field.value}"`);

      if (!field.value && field.type !== 'checkbox') {
        console.log(`ProfVault: Skipping field ${index} - no value`);
        continue;
      }

      let filled = false;

      // Handle custom dropdowns (async - needs await)
      if (field.type === 'custom_dropdown' && field.selector) {
        console.log(`ProfVault: Filling custom dropdown: selector="${field.selector}" value="${field.value}"`);
        filled = await fillCustomDropdown(field.selector, field.value);
        console.log(`ProfVault: Custom dropdown fill result:`, filled);
        if (filled) {
          filledCount++;
        }
        continue;
      }

      // Handle checkbox fields
      if (field.type === 'checkbox') {
        filled = fillCheckbox(field);
        if (filled) {
          filledCount++;
        }
        continue;
      }

      // Handle radio buttons
      if (field.type === 'radio') {
        filled = fillRadio(field);
        if (filled) {
          filledCount++;
        }
        continue;
      }

      // Find the corresponding input for other types
      let input = null;

      // Try to find by ID first
      if (field.id && !field.id.startsWith('field_') && !field.id.startsWith('custom_dropdown_')) {
        input = document.getElementById(field.id);
      }

      // Try to find by name
      if (!input && field.name) {
        try {
          input = document.querySelector(`[name="${field.name}"]`);
        } catch (e) {
          // Name might contain special characters, try CSS.escape if available
          try {
            input = document.querySelector(`[name="${CSS.escape(field.name)}"]`);
          } catch (e2) {
            console.log('ProfVault: Could not find input by name:', field.name);
          }
        }
      }

      // Fall back to index if available
      if (!input && field.index !== undefined) {
        const allInputs = document.querySelectorAll('input:not([type="hidden"]):not([type="submit"]):not([type="button"]):not([type="password"]), textarea, select');
        let idx = 0;
        for (const inp of allInputs) {
          if (idx === field.index) {
            input = inp;
            break;
          }
          idx++;
        }
      }

      if (input) {
        console.log(`ProfVault: Found native input for field ${index}, filling with value="${field.value}"`);
        filled = fillInput(input, field.value, field.type);
        if (filled) {
          filledCount++;
          highlightFilledField(input);
        }
      } else {
        console.log(`ProfVault: Could not find input for field ${index}`);
      }
    }

    console.log(`ProfVault: Finished filling. Total filled: ${filledCount}`);
    return filledCount;
  }

  // Fill a single input with a value
  function fillInput(input, value, fieldType) {
    if (!input || !value) return false;

    try {
      // Handle different input types
      if (input.tagName === 'SELECT' || fieldType === 'select') {
        return fillSelect(input, value);
      }

      if (input.type === 'checkbox') {
        return fillCheckboxInput(input, value);
      }

      if (input.type === 'radio') {
        return fillRadioInput(input, value);
      }

      // For regular inputs and textareas
      input.focus();

      // Clear existing value
      input.value = '';

      // Set the new value
      input.value = value;

      // Trigger events to notify any JS frameworks
      triggerInputEvents(input);

      return true;
    } catch (error) {
      console.error('Error filling input:', error);
      return false;
    }
  }

  // Fill a select element
  function fillSelect(select, value) {
    const valueLower = value.toLowerCase().trim();

    // Helper function to normalize text for comparison
    const normalizeText = (text) => {
      return text.toLowerCase()
        .trim()
        .replace(/[^\w\s]/g, ' ')  // Remove punctuation
        .replace(/\s+/g, ' ')       // Collapse whitespace
        .trim();
    };

    const valueNorm = normalizeText(valueLower);

    // Try exact match first
    for (const option of select.options) {
      if (option.value.toLowerCase() === valueLower ||
          option.text.toLowerCase().trim() === valueLower) {
        select.value = option.value;
        triggerInputEvents(select);
        highlightFilledField(select);
        return true;
      }
    }

    // Try option starts with value (e.g., "Asian" matches "Asian (Not Hispanic or Latino)")
    for (const option of select.options) {
      const optText = option.text.toLowerCase().trim();
      if (optText.startsWith(valueLower + ' ') || optText.startsWith(valueLower + '(')) {
        select.value = option.value;
        triggerInputEvents(select);
        highlightFilledField(select);
        return true;
      }
    }

    // Try normalized match (ignoring punctuation)
    for (const option of select.options) {
      const optNorm = normalizeText(option.text);
      if (optNorm === valueNorm || optNorm.startsWith(valueNorm + ' ')) {
        select.value = option.value;
        triggerInputEvents(select);
        highlightFilledField(select);
        return true;
      }
    }

    // Try partial match
    for (const option of select.options) {
      if (option.text.toLowerCase().includes(valueLower) ||
          valueLower.includes(option.text.toLowerCase().trim())) {
        select.value = option.value;
        triggerInputEvents(select);
        highlightFilledField(select);
        return true;
      }
    }

    // Gender mapping (profile uses "Male"/"Female" but forms use "Man"/"Woman")
    const genderMappings = {
      'male': ['man', 'male', 'm'],
      'female': ['woman', 'female', 'f'],
      'non-binary': ['non-binary', 'nonbinary', 'non binary', 'genderqueer', 'gender non-conforming'],
      'prefer not to say': ['prefer not', 'decline', 'don\'t wish', 'do not wish']
    };

    for (const [profileValue, keywords] of Object.entries(genderMappings)) {
      if (valueLower === profileValue || keywords.includes(valueLower)) {
        for (const option of select.options) {
          const optText = option.text.toLowerCase().trim();
          if (keywords.some(kw => optText.includes(kw) || optText === kw)) {
            select.value = option.value;
            triggerInputEvents(select);
            highlightFilledField(select);
            return true;
          }
        }
      }
    }

    // Special handling for veteran status values
    const veteranMappings = {
      'i am not a veteran': ['i am not', 'not a veteran', 'non-veteran', 'no, i am not'],
      'i am a veteran': ['i am a veteran', 'yes, i am', 'veteran'],
      'i do not wish to answer': ['decline', 'prefer not', 'don\'t wish', 'do not wish']
    };

    for (const [profileValue, keywords] of Object.entries(veteranMappings)) {
      if (valueLower.includes(profileValue) || profileValue.includes(valueLower)) {
        for (const option of select.options) {
          const optText = option.text.toLowerCase().trim();
          if (keywords.some(kw => optText.includes(kw))) {
            select.value = option.value;
            triggerInputEvents(select);
            highlightFilledField(select);
            return true;
          }
        }
      }
    }

    // Special handling for disability status values
    const disabilityMappings = {
      'no, i do not have a disability': ['no, i do not', 'don\'t have a disability', 'do not have a disability'],
      'yes, i have a disability': ['yes, i have', 'have a disability'],
      'i do not wish to answer': ['decline', 'prefer not', 'don\'t wish', 'do not wish']
    };

    for (const [profileValue, keywords] of Object.entries(disabilityMappings)) {
      if (valueLower.includes(profileValue) || profileValue.includes(valueLower)) {
        for (const option of select.options) {
          const optText = option.text.toLowerCase().trim();
          if (keywords.some(kw => optText.includes(kw))) {
            select.value = option.value;
            triggerInputEvents(select);
            highlightFilledField(select);
            return true;
          }
        }
      }
    }

    // Try matching Yes/No variants
    if (['yes', 'true', '1'].includes(valueLower)) {
      for (const option of select.options) {
        const optText = option.text.toLowerCase().trim();
        if (['yes', 'true', '1', 'y'].includes(optText)) {
          select.value = option.value;
          triggerInputEvents(select);
          highlightFilledField(select);
          return true;
        }
      }
    }

    if (['no', 'false', '0'].includes(valueLower)) {
      for (const option of select.options) {
        const optText = option.text.toLowerCase().trim();
        if (['no', 'false', '0', 'n'].includes(optText)) {
          select.value = option.value;
          triggerInputEvents(select);
          highlightFilledField(select);
          return true;
        }
      }
    }

    return false;
  }

  // Helper function to find dropdown menu/listbox after clicking
  function findDropdownMenu(element) {
    let listbox = null;

    // Check aria-controls or aria-owns
    const listboxId = element.getAttribute('aria-controls') || element.getAttribute('aria-owns');
    if (listboxId) {
      listbox = document.getElementById(listboxId);
      if (listbox) return listbox;
    }

    // Look for menu inside the element
    try {
      listbox = element.querySelector('[class*="menu"], [class*="Menu"], [role="listbox"]');
      if (listbox) return listbox;
    } catch (e) {
      console.log('ProfVault: Error finding menu in element');
    }

    // Look for nearby listbox in parent
    try {
      const parent = element.closest('[class*="select"], [class*="dropdown"], [class*="field"]');
      if (parent) {
        listbox = parent.querySelector('[role="listbox"], [class*="menu"], [class*="Menu"], [class*="options"]');
        if (listbox) return listbox;
      }
    } catch (e) {
      console.log('ProfVault: Error finding nearby listbox');
    }

    // Try document-level menu that appeared after click (react-select often appends to body)
    // Look for recently added or visible menus
    try {
      // Greenhouse/React-Select specific patterns
      const menuSelectors = [
        '.select__menu',                           // React-Select
        '[class*="select__menu"]',                 // React-Select variants
        '[class*="menu-list"]',                    // Other libraries
        '[role="listbox"]',                        // ARIA pattern
        '[class*="dropdown-menu"]:not([hidden])', // Bootstrap-style
        '[class*="options-list"]',                // Generic
        '.select-menu'                             // Generic
      ];

      for (const selector of menuSelectors) {
        const menus = document.querySelectorAll(selector);
        for (const menu of menus) {
          // Check if menu is visible
          const style = window.getComputedStyle(menu);
          if (style.display !== 'none' && style.visibility !== 'hidden' && style.opacity !== '0') {
            // Check if menu has options
            const hasOptions = menu.querySelector('[role="option"], [class*="option"], li');
            if (hasOptions) {
              return menu;
            }
          }
        }
      }
    } catch (e) {
      console.log('ProfVault: Could not query for menu elements');
    }

    return null;
  }

  // Fill custom dropdown (div-based)
  async function fillCustomDropdown(selector, value) {
    try {
      // Try to find element by selector, handling potential CSS selector issues
      let element = null;
      try {
        element = document.querySelector(selector);
      } catch (selectorError) {
        console.log('ProfVault: Invalid selector, trying by ID:', selector);
        // If selector fails, try extracting ID from it
        const idMatch = selector.match(/#([a-zA-Z0-9_-]+)/);
        if (idMatch) {
          element = document.getElementById(idMatch[1]);
        }
      }

      if (!element) {
        console.log('ProfVault: Custom dropdown not found:', selector);
        return false;
      }

      if (!value) {
        console.log('ProfVault: No value provided for custom dropdown');
        return false;
      }

      console.log('ProfVault: Filling custom dropdown with value:', value, 'selector:', selector);

      // Find the clickable control area (for react-select style dropdowns)
      let clickTarget = null;
      try {
        // Greenhouse/React-Select patterns
        clickTarget = element.querySelector('[class*="control"], [class*="Control"]');
        if (!clickTarget) {
          clickTarget = element.querySelector('[class*="indicator"], [class*="Indicator"]');
        }
        // Also try Greenhouse's specific patterns
        if (!clickTarget) {
          clickTarget = element.querySelector('.select__control, .select__dropdown-indicator');
        }
      } catch (e) {
        console.log('ProfVault: Error finding click target in dropdown');
      }
      if (!clickTarget) {
        clickTarget = element;
      }

      console.log('ProfVault: Clicking dropdown target:', clickTarget?.className || 'element itself');

      // Click to open the dropdown - try multiple event types for React compatibility
      clickTarget.focus();
      clickTarget.click();
      clickTarget.dispatchEvent(new MouseEvent('mousedown', { bubbles: true, cancelable: true }));
      clickTarget.dispatchEvent(new MouseEvent('mouseup', { bubbles: true, cancelable: true }));

      // Wait for dropdown to open with retry logic (more attempts for slow-loading dropdowns)
      let listbox = null;
      for (let attempt = 0; attempt < 8; attempt++) {
        await new Promise(resolve => setTimeout(resolve, 250));
        listbox = findDropdownMenu(element);
        if (listbox) {
          console.log('ProfVault: Found dropdown menu on attempt', attempt + 1);
          break;
        }
        // Try clicking again if menu didn't appear
        if (attempt === 1 || attempt === 3) {
          console.log('ProfVault: Retrying click on attempt', attempt + 1);
          clickTarget.click();
          clickTarget.dispatchEvent(new MouseEvent('mousedown', { bubbles: true, cancelable: true }));
        }
        // Try clicking the element itself if control didn't work
        if (attempt === 5 && clickTarget !== element) {
          console.log('ProfVault: Trying to click element directly');
          element.click();
        }
      }

      // Find the input field for typing (react-select has a search input)
      let searchInput = null;
      try {
        searchInput = element.querySelector('input[type="text"], input:not([type])');
      } catch (e) {
        console.log('ProfVault: Error finding search input in dropdown');
      }
      if (!searchInput) {
        searchInput = document.activeElement;
        if (searchInput?.tagName !== 'INPUT') {
          searchInput = null;
        }
      }

      // Determine if we should type in the search input
      // Don't type for EEO fields since the profile value might not match option text
      // (e.g., profile has "Male" but options have "Man")
      const valueLowerForCheck = value.toLowerCase().trim();
      const eeoKeywords = [
        'male', 'female', 'man', 'woman', 'non-binary', 'nonbinary',
        'asian', 'black', 'white', 'hispanic', 'latino', 'native', 'pacific',
        'i am not a veteran', 'i am a veteran', 'not a protected veteran', 'protected veteran',
        'no, i do not have a disability', 'yes, i have a disability',
        'no, i don\'t have a disability', 'i don\'t have a disability',
        'i do not have a disability', 'i have a disability',
        'decline', 'prefer not', 'don\'t wish', 'do not wish'
      ];
      const isEeoValue = eeoKeywords.some(eeo =>
        valueLowerForCheck.includes(eeo) || eeo.includes(valueLowerForCheck)
      );

      console.log('ProfVault: Value check - value:', valueLowerForCheck, 'isEEO:', isEeoValue);

      // If there's a search input and NOT an EEO value, type to filter options
      if (searchInput && value && !isEeoValue) {
        console.log('ProfVault: Typing in search input:', value);
        // Clear existing value
        searchInput.value = '';
        searchInput.focus();

        // Type the value character by character for better compatibility
        for (const char of value) {
          searchInput.value += char;
          searchInput.dispatchEvent(new Event('input', { bubbles: true }));
        }

        // Wait for filtering to happen
        await new Promise(resolve => setTimeout(resolve, 300));

        // Re-find the menu after typing (might have changed)
        listbox = findDropdownMenu(element);
      } else if (isEeoValue) {
        console.log('ProfVault: EEO field detected, not typing in search - will browse all options');
      }

      if (listbox && value) {
        const valueLower = value.toLowerCase().trim();
        let options = [];

        // Try multiple strategies to find options
        const optionSelectors = [
          '[role="option"]',
          '[class*="select__option"]',       // React-Select
          '[class*="option"]:not([class*="options"])', // General option class
          '[id*="react-select"][id*="option"]', // React-Select ID pattern
          'li[class*="option"]',
          'div[class*="option"]'
        ];

        try {
          // First try finding options inside the listbox
          for (const selector of optionSelectors) {
            try {
              options = listbox.querySelectorAll(selector);
              if (options.length > 0) {
                console.log('ProfVault: Found options in listbox with selector:', selector);
                break;
              }
            } catch (e) { /* selector might be invalid */ }
          }

          // If no options found in listbox, search document-wide for visible options
          if (options.length === 0) {
            console.log('ProfVault: No options in listbox, searching document-wide...');
            for (const selector of optionSelectors) {
              try {
                const allOptions = document.querySelectorAll(selector);
                const visibleOptions = Array.from(allOptions).filter(opt => {
                  const style = window.getComputedStyle(opt);
                  const rect = opt.getBoundingClientRect();
                  return style.display !== 'none' &&
                         style.visibility !== 'hidden' &&
                         rect.width > 0 && rect.height > 0;
                });
                if (visibleOptions.length > 0) {
                  options = visibleOptions;
                  console.log('ProfVault: Found', options.length, 'visible options document-wide with:', selector);
                  break;
                }
              } catch (e) { /* selector might be invalid */ }
            }
          }
        } catch (e) {
          console.log('ProfVault: Error querying options:', e.message);
          return false;
        }

        console.log('ProfVault: Total options found:', options.length);
        console.log('ProfVault: Looking for value:', valueLower);
        console.log('ProfVault: Available options:', Array.from(options).map(o => o.textContent.trim().substring(0, 50)));

        // Helper function to normalize text for comparison
        const normalizeText = (text) => {
          return text.toLowerCase()
            .trim()
            .replace(/[^\w\s]/g, ' ')  // Remove punctuation
            .replace(/\s+/g, ' ')       // Collapse whitespace
            .trim();
        };

        // Helper function to properly click React-Select options
        const clickOption = (option) => {
          // React-Select needs proper event sequence
          option.dispatchEvent(new MouseEvent('mouseenter', { bubbles: true }));
          option.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
          option.dispatchEvent(new MouseEvent('mousedown', { bubbles: true, cancelable: true }));
          option.dispatchEvent(new MouseEvent('mouseup', { bubbles: true, cancelable: true }));
          option.click();
          // Also try direct click event dispatch
          option.dispatchEvent(new MouseEvent('click', { bubbles: true, cancelable: true }));
        };

        const valueNorm = normalizeText(valueLower);

        // Try exact match first
        for (const option of options) {
          const optionText = option.textContent.trim().toLowerCase();
          if (optionText === valueLower) {
            clickOption(option);
            highlightFilledField(element);
            console.log('ProfVault: Selected exact match:', optionText);
            return true;
          }
        }

        // Try option starts with value (e.g., "Asian" matches "Asian (Not Hispanic or Latino)")
        for (const option of options) {
          const optionText = option.textContent.trim().toLowerCase();
          if (optionText.startsWith(valueLower + ' ') || optionText.startsWith(valueLower + '(')) {
            clickOption(option);
            highlightFilledField(element);
            console.log('ProfVault: Selected starts-with match:', optionText);
            return true;
          }
        }

        // Try normalized match (ignoring punctuation)
        for (const option of options) {
          const optionNorm = normalizeText(option.textContent);
          if (optionNorm === valueNorm || optionNorm.startsWith(valueNorm + ' ')) {
            clickOption(option);
            highlightFilledField(element);
            console.log('ProfVault: Selected normalized match:', option.textContent.trim());
            return true;
          }
        }

        // Try partial match
        for (const option of options) {
          const optionText = option.textContent.trim().toLowerCase();
          if (optionText.includes(valueLower) || valueLower.includes(optionText)) {
            clickOption(option);
            highlightFilledField(element);
            console.log('ProfVault: Selected partial match:', optionText);
            return true;
          }
        }

        // Gender mapping (profile uses "Male"/"Female" but forms use "Man"/"Woman")
        const genderMappings = {
          'male': ['man', 'male', 'm'],
          'female': ['woman', 'female', 'f'],
          'non-binary': ['non-binary', 'nonbinary', 'non binary', 'genderqueer', 'gender non-conforming'],
          'prefer not to say': ['prefer not', 'decline', 'don\'t wish', 'do not wish']
        };

        for (const [profileValue, keywords] of Object.entries(genderMappings)) {
          if (valueLower === profileValue || keywords.includes(valueLower)) {
            for (const option of options) {
              const optText = option.textContent.trim().toLowerCase();
              if (keywords.some(kw => optText.includes(kw) || optText === kw)) {
                clickOption(option);
                highlightFilledField(element);
                console.log('ProfVault: Selected gender match:', optText);
                return true;
              }
            }
          }
        }

        // Special handling for veteran status values
        const veteranMappings = {
          'i am not a veteran': ['i am not', 'not a veteran', 'non-veteran', 'no, i am not'],
          'i am a veteran': ['i am a veteran', 'yes, i am', 'veteran'],
          'i do not wish to answer': ['decline', 'prefer not', 'don\'t wish', 'do not wish']
        };

        for (const [profileValue, keywords] of Object.entries(veteranMappings)) {
          if (valueLower.includes(profileValue) || profileValue.includes(valueLower)) {
            for (const option of options) {
              const optText = option.textContent.trim().toLowerCase();
              if (keywords.some(kw => optText.includes(kw))) {
                clickOption(option);
                highlightFilledField(element);
                console.log('ProfVault: Selected veteran match:', optText);
                return true;
              }
            }
          }
        }

        // Special handling for disability status values
        const disabilityMappings = {
          'no, i do not have a disability': ['no, i do not', 'don\'t have a disability', 'do not have a disability', 'no, i don\'t', 'i do not have'],
          'yes, i have a disability': ['yes, i have', 'have a disability', 'i have a disability'],
          'i do not wish to answer': ['decline', 'prefer not', 'don\'t wish', 'do not wish', 'i don\'t wish']
        };

        // Check if this is a disability-related value
        const isDisabilityValue = valueLower.includes('disability') ||
          valueLower.includes('no, i do not') || valueLower.includes('yes, i have');

        if (isDisabilityValue) {
          console.log('ProfVault: Processing disability value:', valueLower);
          for (const [profileValue, keywords] of Object.entries(disabilityMappings)) {
            if (valueLower.includes(profileValue) || profileValue.includes(valueLower) ||
                keywords.some(kw => valueLower.includes(kw))) {
              for (const option of options) {
                const optText = option.textContent.trim().toLowerCase();
                if (keywords.some(kw => optText.includes(kw))) {
                  clickOption(option);
                  highlightFilledField(element);
                  console.log('ProfVault: Selected disability match:', optText);
                  return true;
                }
              }
            }
          }
        } else {
          // Original logic for non-disability values
          for (const [profileValue, keywords] of Object.entries(disabilityMappings)) {
            if (valueLower.includes(profileValue) || profileValue.includes(valueLower)) {
              for (const option of options) {
                const optText = option.textContent.trim().toLowerCase();
                if (keywords.some(kw => optText.includes(kw))) {
                  clickOption(option);
                  highlightFilledField(element);
                  console.log('ProfVault: Selected disability match:', optText);
                  return true;
                }
              }
            }
          }
        }

        // Try Yes/No matching
        if (['yes', 'true', '1'].includes(valueLower)) {
          for (const option of options) {
            const optText = option.textContent.trim().toLowerCase();
            if (['yes', 'true', '1', 'y'].includes(optText)) {
              clickOption(option);
              highlightFilledField(element);
              return true;
            }
          }
        }

        if (['no', 'false', '0'].includes(valueLower)) {
          for (const option of options) {
            const optText = option.textContent.trim().toLowerCase();
            if (['no', 'false', '0', 'n'].includes(optText)) {
              clickOption(option);
              highlightFilledField(element);
              return true;
            }
          }
        }

        // If still no match, try clicking the first option that's visible
        // (this helps when filtering narrowed it down to one option)
        const visibleOptions = Array.from(options).filter(opt => {
          const style = window.getComputedStyle(opt);
          return style.display !== 'none' && style.visibility !== 'hidden';
        });

        if (visibleOptions.length === 1) {
          clickOption(visibleOptions[0]);
          highlightFilledField(element);
          console.log('ProfVault: Selected only visible option');
          return true;
        }
      }

      // If we got here, no listbox was found or no matching option
      if (!listbox) {
        console.log('ProfVault: Could not find dropdown menu/listbox after clicking. Tried selectors:', selector);
        // Log what we can see in the DOM for debugging
        const visibleMenus = document.querySelectorAll('[class*="menu"], [role="listbox"]');
        console.log('ProfVault: Visible menu-like elements:', visibleMenus.length);
        visibleMenus.forEach((m, i) => {
          console.log(`  Menu ${i}: class="${m.className}", visible=${window.getComputedStyle(m).display !== 'none'}`);
        });
      }

      // Close dropdown if we couldn't fill it - try multiple methods
      closeDropdown(element);
      console.log('ProfVault: Could not fill dropdown, no matching option found for value:', value);
      return false;
    } catch (error) {
      console.error('ProfVault: Error filling custom dropdown:', error.message || error);
      console.error('ProfVault: Error details - selector:', selector, 'value:', value);
      return false;
    }
  }

  // Helper function to close dropdown menus
  function closeDropdown(element) {
    // Try Escape key (works for most React-Select dropdowns)
    document.dispatchEvent(new KeyboardEvent('keydown', { key: 'Escape', bubbles: true }));

    // Also click elsewhere
    document.body.click();

    // Blur the element
    if (element) {
      element.blur();
    }
  }

  // Fill checkbox based on field info
  function fillCheckbox(field) {
    let checkbox = null;

    // Find checkbox by ID or name
    if (field.id && !field.id.startsWith('field_')) {
      checkbox = document.getElementById(field.id);
    }
    if (!checkbox && field.name) {
      try {
        checkbox = document.querySelector(`input[type="checkbox"][name="${field.name}"]`);
      } catch (e) {
        try {
          checkbox = document.querySelector(`input[type="checkbox"][name="${CSS.escape(field.name)}"]`);
        } catch (e2) {
          console.log('ProfVault: Could not find checkbox by name:', field.name);
        }
      }
    }

    if (checkbox) {
      return fillCheckboxInput(checkbox, field.value);
    }

    return false;
  }

  // Fill a checkbox input
  function fillCheckboxInput(checkbox, value) {
    const shouldCheck = ['yes', 'true', '1', 'checked', 'on'].includes(String(value).toLowerCase());

    if (checkbox.checked !== shouldCheck) {
      checkbox.click();
      highlightFilledField(checkbox);
      return true;
    }

    return false;
  }

  // Fill radio button based on field info
  function fillRadio(field) {
    if (!field.name || !field.value) return false;

    // Find all radio buttons in the group
    const radios = document.querySelectorAll(`input[type="radio"][name="${field.name}"]`);
    const valueLower = field.value.toLowerCase().trim();

    for (const radio of radios) {
      const radioValue = radio.value.toLowerCase().trim();
      const radioLabel = findLabel(radio).toLowerCase().trim();

      if (radioValue === valueLower ||
          radioLabel === valueLower ||
          radioLabel.includes(valueLower)) {
        radio.click();
        highlightFilledField(radio);
        return true;
      }
    }

    // Try Yes/No matching
    if (['yes', 'true', '1'].includes(valueLower)) {
      for (const radio of radios) {
        const radioLabel = findLabel(radio).toLowerCase().trim();
        if (['yes', 'true', '1', 'y'].includes(radioLabel) || ['yes', 'true', '1', 'y'].includes(radio.value.toLowerCase())) {
          radio.click();
          highlightFilledField(radio);
          return true;
        }
      }
    }

    if (['no', 'false', '0'].includes(valueLower)) {
      for (const radio of radios) {
        const radioLabel = findLabel(radio).toLowerCase().trim();
        if (['no', 'false', '0', 'n'].includes(radioLabel) || ['no', 'false', '0', 'n'].includes(radio.value.toLowerCase())) {
          radio.click();
          highlightFilledField(radio);
          return true;
        }
      }
    }

    return false;
  }

  // Fill a radio input directly
  function fillRadioInput(radio, value) {
    const valueLower = String(value).toLowerCase().trim();
    const radioValue = radio.value.toLowerCase().trim();

    if (radioValue === valueLower) {
      radio.click();
      highlightFilledField(radio);
      return true;
    }

    return false;
  }

  // Trigger input events for framework compatibility
  function triggerInputEvents(input) {
    // Input event
    input.dispatchEvent(new Event('input', { bubbles: true }));

    // Change event
    input.dispatchEvent(new Event('change', { bubbles: true }));

    // Blur event
    input.dispatchEvent(new Event('blur', { bubbles: true }));

    // For React compatibility
    const nativeInputValueSetter = Object.getOwnPropertyDescriptor(
      window.HTMLInputElement.prototype,
      'value'
    )?.set;

    if (nativeInputValueSetter && input.tagName === 'INPUT') {
      nativeInputValueSetter.call(input, input.value);
      input.dispatchEvent(new Event('input', { bubbles: true }));
    }
  }

  // Highlight filled fields temporarily
  function highlightFilledField(input) {
    const originalBorder = input.style.border;
    const originalBoxShadow = input.style.boxShadow;

    input.style.border = '2px solid #10b981';
    input.style.boxShadow = '0 0 0 3px rgba(16, 185, 129, 0.2)';

    // Remove highlight after 2 seconds
    setTimeout(() => {
      input.style.border = originalBorder;
      input.style.boxShadow = originalBoxShadow;
    }, 2000);
  }

  // Check if page has fillable inputs
  function hasFormInputs() {
    const inputs = document.querySelectorAll('input:not([type="hidden"]):not([type="submit"]):not([type="button"]), textarea, select');
    // Need at least 3 inputs to be considered a form
    return inputs.length >= 3;
  }

  // Create floating button on pages with forms
  function createFloatingButton() {
    // Already exists
    if (document.getElementById('profvault-fill-button')) return;

    // In iframes, only show button if this frame has actual form inputs
    const isInIframe = window !== window.top;

    // Show button if: has forms OR has meaningful inputs (at least 3)
    const hasForms = document.querySelectorAll('form').length > 0;
    const hasInputs = hasFormInputs();

    // For iframes: only show if we have forms or meaningful inputs
    // For top frame: show if has forms or meaningful inputs
    if (!hasForms && !hasInputs) {
      return;
    }

    console.log('ProfVault: Showing button - forms:', hasForms, 'inputs:', hasInputs, 'iframe:', isInIframe);

    const button = document.createElement('div');
    button.id = 'profvault-fill-button';
    button.innerHTML = `
      <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
        <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
      </svg>
    `;

    button.title = 'Auto-fill this form with ProfVault';

    button.addEventListener('click', async () => {
      // Get token from storage
      chrome.storage.local.get(['profvault_token'], async (result) => {
        if (!result.profvault_token) {
          showNotification('Please connect your ProfVault account first', 'error');
          return;
        }

        button.classList.add('loading');
        const apiBaseUrl = APP_SETTINGS.API_BASE_URL;
        const fillResult = await handleFillForm(result.profvault_token, apiBaseUrl);
        button.classList.remove('loading');

        if (fillResult.success) {
          showNotification(`Filled ${fillResult.filledCount} fields!`, 'success');
        } else {
          showNotification(fillResult.error || 'Failed to fill form', 'error');
        }
      });
    });

    document.body.appendChild(button);
  }

  // Show notification toast
  function showNotification(message, type = 'info') {
    const existing = document.getElementById('profvault-notification');
    if (existing) existing.remove();

    const notification = document.createElement('div');
    notification.id = 'profvault-notification';
    notification.className = `profvault-notification ${type}`;
    notification.textContent = message;

    document.body.appendChild(notification);

    // Remove after 3 seconds
    setTimeout(() => {
      notification.classList.add('fade-out');
      setTimeout(() => notification.remove(), 300);
    }, 3000);
  }

  // Initialize floating button after page load
  if (document.readyState === 'complete') {
    createFloatingButton();
  } else {
    window.addEventListener('load', createFloatingButton);
  }
})();
