// ProfVault Extension Background Service Worker
// Uses APP_SETTINGS from config.js

importScripts('../config.js');

const API_BASE_URL = APP_SETTINGS.API_BASE_URL;

// Listen for installation
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    console.log('ProfVault extension installed');
  }

  // Create context menu
  chrome.contextMenus.create({
    id: 'profvault-fill',
    title: 'Fill with ProfVault',
    contexts: ['page', 'editable']
  });
});

// Handle messages from content scripts and popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getToken') {
    getStoredToken().then(token => sendResponse({ token }));
    return true;
  }

  if (request.action === 'verifyToken') {
    verifyToken(request.token).then(result => sendResponse(result));
    return true;
  }

  if (request.action === 'fetchProfile') {
    fetchProfile(request.token).then(result => sendResponse(result));
    return true;
  }

  if (request.action === 'matchFields') {
    matchFields(request.token, request.fields, request.jobContext, request.formHtml).then(result => sendResponse(result));
    return true;
  }
});

// Token storage
async function getStoredToken() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['profvault_token'], (result) => {
      resolve(result.profvault_token || null);
    });
  });
}

// API Functions
async function verifyToken(token) {
  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/auth/verify`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    return await response.json();
  } catch (error) {
    return { success: false, error: error.message };
  }
}

async function fetchProfile(token) {
  try {
    const response = await fetch(`${API_BASE_URL}/api/extension/profile`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    return await response.json();
  } catch (error) {
    return { success: false, error: error.message };
  }
}

async function matchFields(token, fields, jobContext = {}, formHtml = '') {
  console.log('ProfVault BG: matchFields called with', fields.length, 'fields');
  console.log('ProfVault BG: Job context:', jobContext);
  console.log('ProfVault BG: Form HTML length:', formHtml?.length || 0);
  try {
    console.log('ProfVault BG: Fetching from', `${API_BASE_URL}/api/extension/form_fields/match`);
    const response = await fetch(`${API_BASE_URL}/api/extension/form_fields/match`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        fields,
        job_title: jobContext.job_title,
        company: jobContext.company,
        page_url: jobContext.page_url,
        page_title: jobContext.page_title,
        form_html: formHtml
      })
    });
    console.log('ProfVault BG: Response status:', response.status);
    const data = await response.json();
    console.log('ProfVault BG: Response data:', data);
    return data;
  } catch (error) {
    console.error('ProfVault BG: matchFields error:', error);
    return { success: false, error: 'API error: ' + error.message, matched_fields: [] };
  }
}

// Context menu click handler
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'profvault-fill') {
    const token = await getStoredToken();

    if (!token) {
      // Can't show notification from background, just log
      console.log('ProfVault: No token found');
      return;
    }

    // Trigger form fill via content script
    chrome.tabs.sendMessage(tab.id, {
      action: 'fillForm',
      token: token,
      apiBaseUrl: API_BASE_URL
    });
  }
});

// Listen for storage changes to update badge
chrome.storage.onChanged.addListener((changes, namespace) => {
  if (namespace === 'local' && changes.profvault_token) {
    updateBadge();
  }
});

// Badge to show connection status
async function updateBadge() {
  const token = await getStoredToken();

  if (token) {
    const result = await verifyToken(token);
    if (result.success && result.valid) {
      chrome.action.setBadgeText({ text: '' });
    } else {
      chrome.action.setBadgeText({ text: '!' });
      chrome.action.setBadgeBackgroundColor({ color: '#ef4444' });
    }
  } else {
    chrome.action.setBadgeText({ text: '' });
  }
}

// Update badge on startup
updateBadge();
